"""

(c)2014 Martin Tarenskeen <m.tarenskeenATzonnetPOINTnl>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful, 
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""

import array
import os
import zipfile
from tempfile import mkstemp

from . import syxmidi
from . import fourop
from . import fb01
from . import pssx80
from . import vopm
from . import dxcommon
from . import wav2syx
from . import dx9
from . import dx7
from . import korg

MAXFILESIZE = dxcommon.MAXFILESIZE
PROGRAMNAME = "TXconvert"
PROGRAMVERSION = dxcommon.PROGRAMVERSION
PROGRAMDATE = dxcommon.PROGRAMDATE
ALLOWED_CHARACTERS = dxcommon.ALLOWED_CHARACTERS
ACED = fourop.ACED
ACED2 = fourop.ACED2
ACED3 = fourop.ACED3
EFEDS = fourop.EFEDS
DELAY = fourop.DELAY
MID_IN = dxcommon.MID_IN
MID_OUT = dxcommon.MID_OUT

BN4NR=[]
for b in range(4):
    for i in range(25):
        BN4NR.append(str(25*b+i))
    for i in range(7):
        BN4NR.append('--')
for i in range(10):
    BN4NR[i]='0'+BN4NR[i]

########## FILE INPORT/EXPORT #########

def read(infile, offset=0, check=False, yamaha='tx81z', mid_in=MID_IN, mid_out=MID_OUT):
    txdata = []
    channel = 0
    if zipfile.is_zipfile(infile):
        with zipfile.ZipFile(infile, 'r') as zf:
            zflist = zf.namelist()
            print (zflist)
            for n in zflist:
                try:
                    txdat = []
                    d = zf.read(n)
                    tmp = mkstemp()[1]
                    with open(tmp, 'wb') as f:
                        f.write(d)
                    txdat, channel = read2(mid_in, mid_out, tmp, offset, check)
                    os.remove(tmp)
                    txdata += txdat
                    print("\t{}".format(n))
                except:
                    print("WARNING: only {} from {} files in zip were read".format(zflist.index(n)+1, len(zflist)))
                    break
    else:
        txdata, channel = read2(mid_in, mid_out, infile, offset, check, yamaha)
    return txdata, channel


def read2(mid_in, mid_out, infile, offset=0, check=False, yamaha='tx81z'):
    if not os.path.exists(infile):
        print("{}: File not found.".format(infile))
        quit()
    ext = os.path.splitext(infile)[1].lower()
    if MAXFILESIZE == 0:
        size = os.path.getsize(infile)-offset
    else:
        size = min(MAXFILESIZE, os.path.getsize(infile))-offset
    data=array.array("B")
    with open(infile, 'rb') as f:
        data.fromfile(f, offset+size)
    data=data.tolist()[offset:]
    
    FB01 = False
    if yamaha in ('fb01', 'vopm'):
        FB01 = True

    #Dumprequest
    if dxcommon.ENABLE_MIDI:
        if ext == ".req" and data[:2] == [0xf0, 0x43]:
            data, size = dxcommon.req2data(data, mid_in, mid_out)


    data += [0xf7, 0xf7, 0, 0, 0, 0, 0, 0, 0, 0]
    txdata=[]
    fbdata=[]
    channel=0

    #check if file is a MIDI file
    midskip=0
    if data[0:4] == [0x4d, 0x54, 0x68, 0x64]: #"MThd"
        midskip=2


    # Yamaha DX21/TX81Z VMEM
    for i in range(size-4098):
        if data[i]==0xf0 and data[i+1+midskip]==0x43 and data[i+2+midskip]<=0x0f and data[i+3+midskip]==0x04 and data[i+4+midskip]==0x20 and data[i+5+midskip]==0x00 and data[i+6+4097+midskip]==0xf7:
            if (check==False) or (dxcommon.checksum(data[i+6+midskip:i+6+4096+midskip])==data[i+6+4096+midskip]):
                for p in range(32):
                    txdata += data[128*p+i+6+midskip:128*(p+1)+i+6+midskip]
            channel = data[i+2+midskip]

    # Yamaha 4op editbuffer formats
    vcd = []
    for i in range(size-95):
        if data[i]==0xf0 and data[i+1+midskip//2]==0x43 and data[i+2+midskip//2]<=0x0f and data[i+3+midskip//2]==0x03 and data[i+4+midskip//2]==0x00 and data[i+5+midskip//2]==0x5d:
            #VCED
            vcd += data[i+6+midskip//2:i+6+midskip//2+93]
            channel = data[i+2+midskip//2]

    acd = []
    for i in range(size-25):    
        if data[i]==0xf0 and data[i+1+midskip//2]==0x43 and data[i+2+midskip//2]<=0x0f and data[i+3+midskip//2]==0x7e and data[i+4+midskip//2]==0x00 and data[i+6+midskip//2:i+16+midskip//2]==ACED:
            #ACED
            acd += data[i+16+midskip//2:i+16+midskip//2+23]

    acd2 = []
    for i in range(size-12):    
        if data[i]==0xf0 and data[i+1+midskip//2]==0x43 and data[i+2+midskip//2]<=0x0f and data[i+3+midskip//2]==0x7e and data[i+4+midskip//2]==0x00 and data[i+6+midskip//2:i+16+midskip//2]==ACED2:
            #ACED2
            acd2 += data[i+16+midskip//2:i+16+midskip//2+10]

    acd3 = []
    for i in range(size-22):    
        if data[i]==0xf0 and data[i+1+midskip//2]==0x43 and data[i+2+midskip//2]<=0x0f and data[i+3+midskip//2]==0x7e and data[i+4+midskip//2]==0x00 and data[i+6+midskip//2:i+16+midskip//2]==ACED3:
            #ACED3
            acd3 += data[i+16+midskip//2:i+16+midskip//2+20]
    
    efeds = []
    for i in range(size-5):    
        if data[i]==0xf0 and data[i+1+midskip//2]==0x43 and data[i+2+midskip//2]<=0x0f and data[i+3+midskip//2]==0x7e and data[i+4+midskip//2]==0x00 and data[i+6+midskip//2:i+16+midskip//2]==EFEDS:
            #EFEDS
            efeds += data[i+16+midskip//2:i+16+midskip//2+3]

    delay = []
    for i in range(size-4):    
        if data[i]==0xf0 and data[i+1+midskip//2]==0x43 and data[i+2+midskip//2]<=0x0f and data[i+3+midskip//2]==0x7e and data[i+4+midskip//2]==0x00 and data[i+6+midskip//2:i+16+midskip//2]==DELAY:
            #DELAY
            delay += data[i+16+midskip//2:i+16+midskip//2+2]

    count = len(vcd)//93
    if len(delay) != 2*count:
        delay = fourop.initdelay*count
    if len(efeds) != 3*count:
        efeds = fourop.initefeds*count
    if len(acd3) != 20*count:
        acd3 = fourop.initacd3*count
    if len(acd2) != 10*count:
        acd2 = fourop.initacd2*count
    if len(acd) != 23*count:
        acd = fourop.initacd*count
   
    for v in range(count):
        vmm = fourop.vcd2vmm(
            vcd[93*v:93*(v+1)], 
            acd[23*v:23*(v+1)], 
            acd2[10*v:10*(v+1)], 
            acd3[20*v:20*(v+1)], 
            efeds[3*v:3*(v+1)], 
            delay[2*v:2*(v+1)]
            )
        txdata += vmm

    # Yamaha FB01 RAM and ROM voicebanks
    for i in range(size-6360):
        if data[i]==0xf0 and data[i+1+midskip]==0x43 and data[i+2+midskip]==0x75 and data[i+3+midskip]<=0x0f and data[i+6362+midskip]==0xf7:
            for p in range(48):
                fb=[]
                if (check==False) or (dxcommon.checksum(data[i+midskip+76+131*p:i+midskip+76+131*p+128])==data[i+midskip+76+131*p+128]):

                    for n in range(64):
                        fb.append(data[i+midskip+76+131*p+2*n] + (data[i+midskip+76+131*p+2*n+1]<<4))
                    fbdata += fb 

    # Yamaha FB01 voice bank 0
    for i in range(size-6357):
        if data[i]==0xf0 and data[i+1+midskip]==0x43 and data[i+2+midskip]<=0x0f and data[i+3+midskip]==0x0c and data[i+6359+midskip]==0xf7:
            for p in range(48):
                fb=[]
                if (check==False) or (dxcommon.checksum(data[i+midskip+73+131*p:i+midskip+73+131*p+128])==data[i+midskip+73+131*p+128]):
                    for n in range(64):
                        fb.append(data[i+midskip+73+131*p+2*n] + (data[i+midskip+73+131*p+2*n+1]<<4))
                    fbdata += fb

    # Yamaha FB01 instrument 1~8
    for i in range(size-136):
        if data[i]==0xf0 and data[i+1+midskip]==0x43 and data[i+2+midskip]==0x75 and data[i+3+midskip]<=0x0f and (data[i+4+midskip]&8) and data[i+138+midskip]==0xf7:
            fb=[]
            if (check==False) or (dxcommon.checksum(data[i+midskip+9:i+midskip+9+128])==data[i+midskip+9+128]):
                for n in range(64):
                    fb.append(data[i+midskip+9+2*n] + (data[i+midskip+9+2*n+1]<<4))
                fbdata += fb

    # PSS 480/580/680
    for i in range(size-68):
        if data[i]==0xf0 and data[i+1+midskip//2]==0x43 and data[i+2+midskip//2]==0x76 and data[i+3+midskip//2]==0x00 and data[i+midskip//2+71]==0xf7:
            if (check==False) or (dxcommon.checksum(data[i+4+midskip//2:i+4+midskip//2+66])==data[i+4+midskip//2+66]):
                print("WARNING: experimental PSS-480/580/680 2-OP to 4-OP conversion")
                pss=[]    
                for n in range(33):
                    pss.append((data[i+midskip//2+4+2*n]<<4) + data[i+midskip+4+2*n+1])
                txdata += pssx80.pss2vmm(pss, infile)

    # Yamaha DX9 sysex
    # Single voice
    if size>160:
        for i in range(size-157):
            if data[i]==0xf0 and data[i+1+midskip]==0x43 and data[i+2+midskip]<=0x0f and data[i+3+midskip]==0 and data[i+4+midskip]==1 and data[i+5+midskip]==0x1b and data[i+6+156+midskip]==0xf7:
                if (check==False) or (dxcommon.checksum(data[i+6+midskip:i+6+midskip+155])==data[i+6+midskip+155]):
                    dx9data = dx7.vced2vmem(data[i+6+midskip:i+6+midskip+155])
                else:
                    dx9data = dx7.initvmem
                txdata += dx9.dx9to4op(dx9data)
                channel=data[i+2+midskip]

    # Yamaha DX9 sysex
    # 32-voice (20-voice) bank
    if size>4101:
        for i in range(size-4098):
            #data[i+4+midskip] should be 0x20 but often is 0x10 
            if data[i]==0xf0 and data[i+1+midskip]==0x43 and data[i+2+midskip]<=0x0f and data[i+3+midskip]==0x09 and (data[i+4+midskip]==0x10 or data[i+4+midskip]==0x20) and data[i+5+midskip]==0x00 and data[i+6+4097+midskip]==0xf7:
                if (check==False) or (dxcommon.checksum(data[i+midskip+6:i+midskip+6+4096]) == data[i+midskip+6+4096]):
                    dx9data = data[i+midskip+6:i+midskip+6+4096]
                else:
                    dx9data = dx7.initvmem*32
                for n in range(len(dx9data)//128):
                    txdata += dx9.dx9to4op(dx9data[128*n:128*(n+1)])
                channel = data[i+2+midskip]

    # KORG 707 KORG DS8 7985-550=7435
    for i in range(size-7435): #100 programs bank
        if data[i]==0xf0 and data[i+1+midskip]==0x42 and (data[i+2+midskip] & 0xf0)==0x30 and (data[i+3+midskip] in (0x1a, 0x10)) and data[i+4+midskip]==0x4c:
            print("TODO: Korg 707/DS8 import")
            korgmid = data[i+5+midskip:i+5+midskip+7984]
            korgdat = korg.korg7to8(korgmid)
            for i in range(100):
                txdata += korg.bnk2vmm(korgdat[66*i:66*(i+1)])

    for i in range(size-97): #1 program
        if data[i]==0xf0 and data[i+1+midskip]==0x42 and (data[i+2+midskip] & 0xf0)==0x30 and (data[i+3+midskip] in (0x1a, 0x10)) and data[i+4+midskip]==0x40:
            print("TODO: Korg 707/DS8 import")
            korgmid = data[i+5+midskip:i+5+midskip+96]
            korgdat = korg.korg7to8(korgmid)
            txdata += korg.vce2vmm(korgdat)

    # Still not found anything? Then keep searching ...
    if txdata != []:
        pass

    #Voyetra SIDEMAN and PATCHMASTER
    elif data[0:4] == [0xdf, 0x05, 0x01, 0x00]:
        if ext == ".b15" and size == 10755:
            if dxcommon.list2string(data[6:18]) == "YAMAHA TX81Z":
                for i in range(32):
                    txdata += data[0x060f+128*i:0x60f+128*(i+1)]
        if ext == ".b16" and size == 20837:
            if dxcommon.list2string(data[6:18]) == "YAMAHA FB-01":
                for bank in range(2):
                    for i in range(48):
                        fb = []
                        for p in range(64):
                            lo = data[0x2029*bank + 0x0821 + 130*i + 2*p]&15
                            hi = (data[0x2029*bank + 0x0821 + 130*i + 2*p + 1] & 15) << 4
                            fb.append(lo + hi)
                        fbdata += fb

    # X-OR V50
    elif size == 24576 and ext == ".v5b":
        for i in range(100):
            txdata += data[128*i:128*(i + 1)]

    # X-OR TX81Z
    elif size == 6538 and ext == ".txz":
        for i in range(32):
            txdata += data[128*i:128*(i + 1)]

    # V50 internal disk SYN format
    elif size == 26624 and ext[:-2]==".i":
        for i in range(100):
            d = data[13824 + 128*i:13824 + 128*(i+1)]
            if d[86]<50:
                d[86] += 51
            else:
                d[86] -= 50
            txdata += d

    # V50 YS MCD32 V50 disk
    elif size == 32768 and ext[:-2] == ".c":
        if data[4:10] == [0x59, 0x53, 0x20, 0x53, 0x2f, 0x56]: #"YS S/V"
            for b in range(4):
                for p in range(25):
                    d = data[0x1000 + 0x2000*b + 128*p : 0x1000 + 0x2000*b + 128*(p+1)]
                    if d[86]<50:
                        d[86] += 51
                    else:
                        d[86] -= 50
                    txdata += d
        elif data[4:10] == [0x56, 0x35, 0x30, 0x53, 0x59, 0x4e]: #"V50SYN"
            for b in range(4):
                for p in range(25):
                    d = data[0x3900 + 0xc80*b + 128*p : 0x3900 + 0xc80*b + 128*(p+1)]
                    if d[86]<50:
                        d[86] += 51
                    else:
                        d[86] -= 50
                    txdata += d


    # m.gregory's TX81Z Programmer .tx8
    elif ext == '.tx8' and data[:5] == [0x54, 0x58, 0x38, 0x31, 0x5a]:        
        for i in range((size-9)//173):
            txdata += data[54 + 173*i:54 + 173*i + 128]

    # Emagic Sounddiver TX81Z
    elif ext == ".lib" and data[:4] == [0x4d, 0x52, 0x4f, 0x46]:
        for i in range(size-158):
            if data[i:i+4] == [0x54, 0x4e, 0x45, 0x4c]:
                nmlen=data[i+21]
                if data[i+nmlen+22:i+nmlen+29] == [0x03, 0x7c, 0x54, 0x58, 0x38, 0x31, 0x5a]:
                    vmm = fourop.initvmm
                    vmm[:50] = data[i+nmlen+30:i+nmlen+30+50]
                    vmm[57:67] = data[i+22:i+22+nmlen] + [32]*(10-nmlen)
                    vmm[67:100] = data[i+nmlen+30+57:i+nmlen+30+90]
                    txdata += vmm
                   
    # Synthworks 4OP
    elif ext == '.bnk' and size == 4032 and data[0] == 0x0a:
        for i in range(32):
            txdata += fourop.steinb2vmm(data[1 + 12*i:11 + 12*i], data[0x180 + 114*i:0x180 + 114*(i+1)])
    elif ext == '.snd' and size == 126 and data[0] == 0x0a:
        txdata += fourop.steinb2vmm(data[1:11], data[12:])
 
    # Atari PSSED pss480-780 library with 100 patches
    elif ext == ".lib" and size == 4908:
        print("WARNING: experimental PSS-480/580/680 2-OP to 6-OP conversion")
        for i in range(100):
            voicename = dxcommon.list2string(data[8+16*i:8+16*(i+1)])
            txdata += pssx80.pss2vmm(data[1608+33*i:1608+33*(i+1)], infile, voicename)

    # Dr.T FB01 / raw data
    elif (size % 64 == 0) and ext == ".fb1":
        for i in range(48):
            fbdata += data[64*i:64*(i+1)]

    # Cages Artist FB01
    elif size == 8768 and ext == ".fb1":
        for bnk in range(2):
            for i in range(48):
                fbdata += data[32 + 3104*bnk + 64*i:32 + 3104*bnk + 64*(i+1)]

    # Synthworks FB01
    elif size == 2450 and ext == ".fbd":
        for i in range(49):
            fb = 64*[0]
            fb[:7] = data[1+8*i:8+8*i]
            fb[8:48] = data[0x188 + 42*i:0x1b0 + 42*i]
            fb[58:60] = data[0x1b0 + 42*i:0x1b2 + 42*i]
            fbdata += fb

    # Frederic Meslin FB01 editor
    elif size == 354 and ext == ".fbv":
        #fbdata += data[0x107:0x107+64]
        pass

    # CX5M VOG
    elif size in (3111, 3079) and ext == ".vog":
        fbdata += data[0x27:0x27 + 48*64]

    # VOPM .OPM
    elif ext == ".opm":
        print("WARNING: experimental VOPM (.opm) to 4-OP DX/TX conversion")
        data = vopm.file2data(infile)
        for i in range(len(data)//76):
            fb = vopm.vopm2fb01(data[76*i:76*(i+1)])
            if FB01:
                fbdata += fb
            else:
                tx = fb01.fb2vmm(fb, yamaha)
                tx[57:67] = data[76*i:76*i+10]
                txdata += tx

    # datacassete data from DX21/27/100/11/TX81Z 
    # extracted using Martin Ward's tape-read perl script with baud=1200
    elif ext == ".txt" and size>75:
        txdata += wav2syx.cas2vmm(data, 0)

    # datacassette data extracted using wav2cas with -p option
    elif data[:8] == [0x1f, 0xa6, 0xde, 0xba, 0xcc, 0x13, 0x7d, 0x74]:
        for i in range(len(data)-84):
            if data[i:i+8] == [0x1f, 0xa6, 0xde, 0xba, 0xcc, 0x13, 0x7d, 0x74]:
                txdata += wav2syx.cas2vmm(data, i+8)

    # WAV data with datacassette dump
    elif ext == '.wav':
       txdata += wav2syx.wav2syx(infile, 'tx')
 
    # Dr.T FourOp de Luxe or raw headerless sysex
    elif size % 128 == 0:
        for i in range(size//128):
            if sum(data[128*i + 118:128*i + 128])==0 and sum(data[128*i:128*i + 118]) != 0:
                txdata += data[128*i:128 + 128*i]

    # Audiogrill TX81Z/DX11 Voice Archive 
    elif dxcommon.list2string(data[:26]) == "* TX81Z/DX11 Voice Archive":
        numbers = []
        hexnum = '0123456789abcdef'
        with open(infile, 'r') as f:
            lines = f.readlines()
        for ll in lines:
            if ll[:1] in hexnum:
                for l in ll:
                    if l in hexnum:
                        numbers.append(hexnum.index(l))
        for i in range(len(numbers)//2):
            txdata.append (16*numbers[2*i] + numbers[2*i+1]) 

    # Tim Thompson's Glib tx81z dx100 editor
    elif data[0] == 0xdd and (ext in ('.dx1', '.tx8')) and size==4097:
        txdata += data[1:4097]

    # Ignore many other SysEx
    elif ext in ('.dx7', '.tx7', '.dx2', '.dxc', '.b13', '.b68', '.slb', '.xal', '.dxl'):
        txdata += fourop.initvmm

    # FB to TX / TX to FB
    if not FB01:
        if fbdata != []:
            print("Warning: FB01 to 4-OP DX/TX conversion")
            for i in range(len(fbdata)//64):
                txdata += fb01.fb2vmm(fbdata[64*i:64*(i+1)])
    else:
        if txdata != []:
            print("Warning: 4-OP DX/TX to FB01 conversion")
            for i in range(len(txdata)//128):
                fbdata += fb01.vmm2fb(txdata[128*i:128*(i+1)])
    
    if not FB01:
        if txdata == []:
            for i in data[:128]:
                if i > 127:
                    txdata += fourop.initvmm
                    break

    # If everything fails: The worst thing that can happen is
    # creating a TX/DX compatible but completely useless patchbank :-)
    # Should work with e.g. headerless 4096 bytes DXX files
    if fbdata == [] and txdata == []:
        if FB01:
            fbdata += data[:64*(len(data)//64)]
        else:
            txdata += data[:128*(len(data)//128)]

    # Safety garbage cleanup
    txdat = []
    initcount = 0
    if FB01:
        for i in range(len(fbdata)//64):
            if dxcommon.list2string(fbdata[64*i:64*i+5]).lower() != "initv":
                if dxcommon.list2string(fbdata[64*i:64*i+7]) != "SineWav":
                    if dxcommon.list2string(fbdata[64*i:64*i+7]).strip() != "":
                        txdat += fb01.fbclean(fbdata[64*i:64*(i+1)])
        #txdat += fb01.initfb*(48 - ((len(txdat)//64) % 48))
    else:
        for i in range(len(txdata)//128):
            if dxcommon.list2string(txdata[128*i+57:128*i+67]) != "INIT VOICE":
                if dxcommon.list2string(txdata[128*i+57:128*i+67]).strip() != "":
                    txdat += fourop.cleanvmm(txdata[128*i:128*(i+1)])
        #txdat += fourop.initvmm*(32 - ((len(txdat)//128) % 32))

    return txdat, channel

###### TX WRITE STUFF #####

def write_fb(outfile, fbdata, channel, nosplit, split=48, yamaha='fb01', mid_out='MID_OUT'):
    FB01 = True
    ext = os.path.splitext(outfile)[1].lower()
    if ext == ".opm":
        split = 128
        yamaha = "vopm"
    basename = os.path.split(outfile)[1]
    basename = os.path.splitext(basename)[0]
    dirname = os.path.dirname(outfile)
    mid = False
    if ext.lower() in (".mid", ".midi"):
        mid = True
    if dxcommon.ENABLE_MIDI:
        if outfile[-4:] == "MIDI":
            nosplit = True 
 
    if len(fbdata)>64:
        patchcount = len(fbdata)//64
    elif len(fbdata)<64:
        fbdata = fb01.initfb
        patchcount = 1
    else:
        patchcount = 1
    
    
    if len(fbdata)>64:
        #Fill unused patch locations in banks with INIT VOICE
        while len(fbdata)%3072 != 0:
            fbdata += fb01.initfb

        patchcount = len(fbdata)//64
    

    if split == 1:
        patchcount = 1

    dat=[]
    message = ''

    bankcount = max(1, len(fbdata)//(64*split))
    bankdigits = len(str(bankcount+1))
    for bank in range(bankcount):
        if len(fbdata)//(64*split)>1 and nosplit==False:
            outfile=os.path.join(dirname, basename+"("+str(bank+1).zfill(bankdigits)+")"+ext)
        if nosplit==False:
            dat=[]

        if ext.lower() == ".fb1":
            if patchcount == 1:
                dat = fbdata
            else:
                dat += fbdata[3072*bank:3072*(bank+1)]
        
        elif ext.lower() == ".txt":
            dat = fb01.fb2txt(fbdata)

        elif ext.lower() == ".opm":
            dat += vopm.fb2vopm(fbdata[8192*bank:8192*(bank+1)])
        
        else:
            if patchcount == 1: #save as VCED sysex
                dat += fb01.fb2syx(fbdata[:64], bank%2, channel)
                
            else: #save as VMEM sysex
                dat += fb01.fb2syx(fbdata[3072*bank:3072*(bank+1)], bank%2, channel)

        if nosplit==False:
            if mid:
                dat = syxmidi.syx2mid(dat)
            with open(outfile, 'wb') as f:
                array.array('B', dat).tofile(f)
            print("{} Patch(es) written to: {}".format(min(split, len(fbdata)//64), outfile))

    if nosplit==True:
        if mid:
            dat = syxmidi.syx2mid(dat)
        with open(outfile, 'wb') as f:
            array.array('B', dat).tofile(f)
        print("{} Patch(es) written to: {}".format(len(fbdata)//64, outfile))

    if dxcommon.ENABLE_MIDI:
        if outfile[-4:] == "MIDI":
            dxcommon.data2midi(dat, mid_out)

    return "Ready. {} Patch(es) written to output file(s)".format(len(fbdata)//64)



def write(outfile, txdata, channel=0, nosplit=False, split=32, yamaha='tx81z', mid_out='MID_OUT'):
    ext = os.path.splitext(outfile)[1]
    if ext.lower() == '.opm':
        yamaha = 'vopm'
        split = 128
    if ext.lower() == '.fb1':
        yamaha = 'fb01'
        if split !=1: split = 48
    if yamaha in ('fb01', 'vopm'):
        return write_fb(outfile, txdata, channel, nosplit, split, yamaha, mid_out)

    basename = os.path.split(outfile)[1]
    basename = os.path.splitext(basename)[0]
    dirname = os.path.dirname(outfile)

    mid = False
    if ext.lower() in (".mid", ".midi"):
        mid = True
    if dxcommon.ENABLE_MIDI:
        if outfile[-4:] == "MIDI":
            nosplit = True 
 
    if len(txdata)>128:
        patchcount = len(txdata)//128
    elif len(txdata)<128:
        txdata = fourop.initvmm
        patchcount = 1
    else:
        patchcount = 1

    if split == 0:
        nosplit = True

    elif patchcount>1:
        if patchcount <= 32:
            split = 32
        txdat = []
        txrange = len(txdata)/float(split*128)
        if txrange > int(txrange):
            txrange += 1
        txrange = int(txrange)
        for i in range(txrange):
            txdat += txdata[128*i*split:128*(i+1)*split] + fourop.initvmm*(32-split)
        txdata = txdat
    
    if yamaha in ('ys200', 'ys100', 'tq5', 'b200'):
        for i in range(len(txdata)//128):
            txdata[128*i:128*(i+1)] = fourop.v50_ys(txdata[128*i:128*(i+1)])

    if yamaha == ('v50'):
        for i in range(len(txdata)//128):
            txdata[128*i:128*(i+1)] = fourop.ys_v50(txdata[128*i:128*(i+1)])

    # Operator frequency recalculation on older models
    if yamaha in ('dx100', 'dx27', 'dx27s', 'dx21'):
        for i in range(len(txdata)//128):
            txdata[128*i:128*i+128] = fourop.tx81z_dx21(txdata[128*i:128*i+128])[0]
                
    if yamaha in ('v50', 'ys200', 'ys100', 'tq5', 'ds55', 'b200') and len(txdata)>4096:
        bn=4
    else:
        bn=1
    
    if len(txdata)>128:
        #Fill unused patch locations in banks with INIT VOICE
        while len(txdata)%(4096*bn) != 0:
            txdata += fourop.initvmm
        patchcount = len(txdata)//128

    dat=[]
    message = ''

    bankcount = max(1, len(txdata)//(bn*4096))
    bankdigits = len(str(bankcount+1))
    for bank in range(bankcount):
        if len(txdata)//(bn*4096)>1 and nosplit==False:
            outfile=os.path.join(dirname, basename+"("+str(bank+1).zfill(bankdigits)+")"+ext)
        if nosplit==False:
            dat=[]

        if ext.lower() == ".dxx":
            if patchcount == 1:
                dat = txdata[:128]
            else:
                dat += txdata[4096*bn*bank:4096*bn*(bank+1)]

        elif ext.lower() == ".txt":
            if patchcount == 1:
                dat = fourop.vmm2txt(txdata[:128], yamaha)
            else:
                t = ''
                for i in range(32*bn):
                    if bn == 4:
                        bnr = BN4NR[i%128]
                    else:
                        bnr = 1+i+bank*32*bn
                    t += "{} : {}\n".format(bnr, fourop.voicename(txdata[4096*bank*bn+128*i:4096*bank*bn+128*(i+1)]))
                t += "\n"

                for i in t:
                    dat.append(ord(i))

        else:
            if patchcount == 1: #save as VCED sysex
                vcd, acd, acd2, acd3, efeds, delay = fourop.vmm2vcd(txdata) 
                
                #ACED3
                if yamaha in ('v50', 'all'):
                    dat += [0xf0, 0x43, channel, 0x7e, 0x00, 0x1e]
                    dat += ACED3
                    dat += acd3
                    dat += [dxcommon.checksum(acd3+ACED3), 0xf7]
                
                #EFEDS
                if yamaha in ('ys100', 'ys200', 'b200', 'tq5', 'all'):
                    dat += [0xf0, 0x43, channel, 0x7e, 0x00, 0x0d]
                    dat += EFEDS
                    dat += efeds
                    dat += [dxcommon.checksum(efeds+EFEDS), 0xf7]
                
                #DELAY
                if yamaha in ('ds55', 'all'):
                    dat += [0xf0, 0x43, channel, 0x7e, 0x00, 0x0c]
                    dat += DELAY
                    dat += delay
                    dat += [dxcommon.checksum(delay+DELAY), 0xf7]
                
                #ACED2
                if yamaha in ('dx11', 'v50', 'ys100', 'ys200', 'tq5', 'b200', 'wt11', 'all'):
                    dat += [0xf0, 0x43, channel, 0x7e, 0x00, 0x14]
                    dat += ACED2
                    dat += acd2
                    dat += [dxcommon.checksum(acd2+ACED2), 0xf7]
                
                #ACED
                if yamaha in ('ds55', 'dx11', 'v50', 'ys100', 'ys200', 'tq5', 'b200', 'tx81z', 'wt11', 'all'):
                    dat += [0xf0, 0x43, channel, 0x7e, 0x00, 0x21]
                    dat += ACED
                    dat += acd
                    dat += [dxcommon.checksum(acd+ACED), 0xf7]

                #VCED
                dat += [0xf0, 0x43, channel, 0x03, 0x00, 0x5d]
                dat += vcd
                dat += [dxcommon.checksum(vcd), 0xf7]
                
                
            else: #save as VMEM sysex
                if bn == 4:
                    for block in range(bn):
                        dat += [0xf0, 0x43 , 0x10+channel, 0x24, 0x07, 1+block, 0xf7]
                        dat += [0xf0, 0x43, channel, 0x04, 0x20, 0x00]
                        dat += txdata[4096*block+16384*bank:4096*(block+1)+16384*bank]
                        dat += [dxcommon.checksum(txdata[4096*block+16384*bank:4096*(block+1)+16384*bank]), 0xf7]
                if bn == 1:
                    dat += [0xf0, 0x43, channel, 0x04, 0x20, 0x00]
                    dat += txdata[4096*bank:4096*(bank+1)]
                    dat += [dxcommon.checksum(txdata[4096*bank:4096*(bank+1)]), 0xf7]

        if nosplit==False:
            if mid:
                dat = syxmidi.syx2mid(dat)
            with open(outfile, 'wb') as f:
                array.array('B', dat).tofile(f)
            print("{} Patch(es) written to: {}".format(min(32*bn, patchcount), outfile))

    if nosplit==True:
        if mid:
            dat = syxmidi.syx2mid(dat)
        with open(outfile, 'wb') as f:
            array.array('B', dat).tofile(f)
        print("{} Patch(es) written to: {}".format(patchcount, outfile))

    if dxcommon.ENABLE_MIDI:
        if outfile[-4:] == "MIDI":
            dxcommon.data2midi(dat, mid_out)

    return "Ready. {} Patch(es) written to output file(s)".format(len(txdata)//128)

def tx2list(txdata):
    txlist = []
    for i in range(len(txdata)//128):
        txdat = txdata[128*i:128*(i+1)]
        a = txdat[57:67]+txdat[:57]+txdat[67:]
        txlist.append(a)
    return txlist

def fb2list(fbdata):
    fblist = []
    for i in range(len(fbdata)//64):
        fblist.append(fbdata[64*i:64*(i+1)])
    return fblist

def list2tx(data):
    txdata = []
    for i in data:
        txdata += i[10:67]+i[:10]+i[67:]
    return txdata

def list2fb(data):
    fbdata = []
    for i in data:
        fbdata += i
    return fbdata

def txsort(txdata, casesens=False):
    if (len(txdata)//128) > 1:
        print("Sorting patches by name ...")
        l = tx2list(txdata)
        L = []
        for n in range(len(l)):
            if dxcommon.list2string(l[n][:10]) != "INIT VOICE":
                L.append(l[n])
        if casesens:
            L.sort()
        else:
            for i in range(len(L)):
                L[i] = dxcommon.list2string(L[i])
            L.sort(key=str.lower)
            for i in range(len(L)):
                L[i] = dxcommon.string2list(L[i])
        txdata = list2tx(L)
    return txdata

def fbsort(fbdata, casesens=True):
    if (len(fbdata)//64) > 1:
        print("Sorting patches by name ...")
        l = fb2list(fbdata)
        L = []
        for n in range(len(l)):
            if dxcommon.list2string(l[n][:6]) != "InitVc":
                L.append(l[n])
        if casesens:
            L.sort()
        else:
            for i in range(len(L)):
                L[i] = dxcommon.list2string(L[i])
            L.sort(key=str.lower)
            for i in range(len(L)):
                L[i] = dxcommon.string2list(L[i])
        fbdata = list2fb(L)
    return fbdata

def txnodupes(txdata, nodupes2):
    if (len(txdata)//128) > 1:
        print("Searching and removing duplicates ...")
        L = tx2list(txdata)
        l=[L[0]]
        if nodupes2 == True:
            for i in range(len(L)):
                dup = False
                for j in range(len(l)):
                    if l[j][10:] == L[i][10:]:
                        dup=True
                        break
                if dup == False:
                    l.append(L[i])
        else:
            for i in L:
                if not i in l:
                    l.append(i)
        txdata = list2tx(l)
    return txdata

def fbnodupes(fbdata, nodupes2):
    if (len(fbdata)//64) > 1:
        print("Searching and removing duplicates ...")
        L = fb2list(fbdata)
        l=[L[0]]
        if nodupes2 == True:
            for i in range(len(L)):
                dup = False
                for j in range(len(l)):
                    if l[j][7:] == L[i][7:]:
                        dup=True
                        break
                if dup == False:
                    l.append(L[i])
        else:
            for i in L:
                if not i in l:
                    l.append(i)
        fbdata = list2fb(l)
    return fbdata

def txfind(txdata, keywords):
    print("Searching for {}".format(keywords))
    txdat = []
    keywords = keywords.split(",")
    for keyword in keywords:
        for i in range(len(txdata)//128):
            s = ''
            for k in range(128*i+57, 128*i+67):
                s += chr(txdata[k])
            if keyword.lower() in s.lower():
                txdat += txdata[128*i:128*i+128]
    return txdat

def fbfind(fbdata, keywords):
    print("Searching for {}".format(keywords))
    fbdat = []
    keywords = keywords.split(",")
    for keyword in keywords:
        for i in range(len(fbdata)//64):
            s = ''
            for k in range(64*i, 64*i+7):
                s += chr(fbdata[k])
            if keyword.lower() in s.lower():
                fbdat += fbdata[64*i:64*(i+1)]
    return fbdat

def txrandom(txdata, deviation):
    print('Randomizing data ...')
    import random
    n = len(txdata)//128

    if deviation == 0:
        # thanks to Renata Dokmanovic 
        newdata = []+txdata
        for patch in range(n):
            for param in range(101):
                newdata[128 * patch + param] = txdata[128 * random.randrange(n) + param]
        return newdata

    newdata = []
    pmax = [0]*128
    pmin = [127]*128
    deviation = min(deviation, 300)
    dev = deviation%100

    for p in range(128):
        for i in range(n):
            if txdata[128*i+p] > pmax[p]:
                pmax[p] = txdata[128*i+p]
            if txdata[128*i+p] < pmin[p]:
                pmin[p] = txdata[128*i+p]

    for i in range(n):
        for p in range(128):
            pmax[p] = txdata[128*i+p] + int(round((pmax[p]-txdata[128*i+p])*(dev/100.)))
            pmin[p] = txdata[128*i+p] - int(round((txdata[128*i+p]-pmin[p])*(dev/100.)))
            newdata.append(random.randint(pmin[p], pmax[p]))
        
        alg = txdata[128*i+40]&7
        algnew = newdata[128*i+40]&7
        if deviation < 201: # don't randomize algo
            newdata[128*i+40] = alg + (newdata[128*i+40]&0b1111000)
            algnew = alg
        
        # keep carrier frequencies playable
        newdata[128*i+70:128*i+73] = txdata[128*i+70:128*i+73] #PEG levels
        newdata[128*i+46] = txdata[128*i+46] #Transpose
        for op in range(4):
            Op=(0, 2, 1, 3)[op]
            if fourop.carrier(algnew, op) and fourop.carrier(alg, op):
                for p in (8, 9):
                    newdata[128*i+10*(3-Op)+p] = txdata[128*i+10*(3-Op)+p]
                for p in (73, 74):
                    newdata[128*i+2*(3-Op)+p] = txdata[128*i+2*(3-Op)+p]
            elif fourop.carrier(algnew, op): # use OP1 frequency
                for p in (8, 9):
                    newdata[128*i+10*(3-Op)+p] = txdata[128*i+10*(3-0)+p]
                for p in (73, 74):
                    newdata[128*i+2*(3-Op)+p] = txdata[128*i+2*(3-0)+p]
        
        if deviation < 101: # don't randomize modulator frequencies either
            for op in range(4):
                Op=(0, 2, 1, 3)[op]
                if not fourop.carrier(alg, op):
                    for p in (8, 9):
                        newdata[128*i+10*(3-Op)+p] = txdata[128*i+10*(3-Op)+p]
                    for p in (73, 74):
                        newdata[128*i+2*(3-Op)+p] = txdata[128*i+2*(3-Op)+p]

        # auto create nice patchnames, 
        #reflecting the deviation value
        if (newdata[128*i:128*i+57] == txdata[128*i:128*i+57]) and (newdata[128*i+67:128*i+101] == txdata[128*i+67:128*i+101]):
            newdata[128*i+57:128*i+67] == txdata[128*i+57:128*i+67]
        else:
            for p in range(57, 67):
                if (p-57) < (10-dev//10):
                    newdata[128*i+p] = txdata[128*i+p]
                else:
                    newdata[128*i+p] = random.randrange(65, 91)
    return newdata

def fbrandom(fbdata, deviation):
    print('Randomizing data ...')
    import random
    n = len(fbdata)//64

    if deviation == 0:
        # thanks to Renata Dokmanovic 
        newdata = []+fbdata
        for patch in range(n):
            for param in range(64):
                newdata[64 * patch + param] = fbdata[64 * random.randrange(n) + param]
        return newdata

    newdata = []
    pmax = [0]*64
    pmin = [255]*64
    deviation = min(deviation, 300)
    dev = deviation%100

    for p in range(64):
        for i in range(n):
            if fbdata[64*i+p] > pmax[p]:
                pmax[p] = fbdata[64*i+p]
            if fbdata[64*i+p] < pmin[p]:
                pmin[p] = fbdata[64*i+p]

    for i in range(n):
        for p in range(64):
            pmax[p] = fbdata[64*i+p] + int(round((pmax[p]-fbdata[64*i+p])*(dev/100.)))
            pmin[p] = fbdata[64*i+p] - int(round((fbdata[64*i+p]-pmin[p])*(dev/100.)))
            newdata.append(random.randint(pmin[p], pmax[p]))
        
        alg = fbdata[64*i+12]&7
        algnew = newdata[64*i+12]&7
        if deviation < 201: # don't randomize algo/feedback
            newdata[64*i+12] = alg + (newdata[64*i+12]&0b111000)
            algnew = alg
        
        # keep carrier frequencies playable
        newdata[64*i+15] = fbdata[64*i+15] #Transpose
        for op in range(4):
            opad = (40, 32, 24, 16)[op]
            if fourop.carrier(algnew, op) and fourop.carrier(alg, op):
                for p in (3, 6): #crs, det
                    newdata[64*i + opad + p] = fbdata[64*i + opad + p]
            elif fourop.carrier(algnew, op): # use OP1 frequency
                for p in (3, 6):
                    newdata[64*i + opad + p] = fbdata[64*i + opad + p]
        
        if deviation < 101: # don't randomize modulator frequencies either
            for op in range(4):
                opad = (40, 32, 24, 16)[op]
                if not fourop.carrier(alg, op):
                    for p in (3, 6):
                        newdata[64*i + opad + p] = fbdata[64*i + opad + p]

        # auto create nice patchnames, 
        #reflecting the deviation value
        if (newdata[64*i + 8:64*i + 64] == fbdata[64*i + 8:64*i + 64]):
            newdata[64*i:64*i + 7] == fbdata[64*i:64*i + 7]
        else:
            for p in range(7):
                if p < (10-dev//10):
                    newdata[64*i + p] = fbdata[64*i + p]
                else:
                    newdata[64*i + p] = random.randrange(65, 91)
    return newdata

def txswap(a, b, txdata):
    print("Swapping patches ...")
    adata = txdata[128*(a-1):128*a]
    bdata = txdata[128*(b-1):128*b]
    txdata[128*(a-1):128*a] = bdata
    txdata[128*(b-1):128*b] = adata
    return txdata

def txcopy(a, b, txdata):
    print("Copying patch(es) ...")
    for i in a:
        adata = txdata[128*(i-1):128*i]
        txdata[128*(b-1):128*b] = adata
        b += 1
    return txdata

def fbswap(a, b, fbdata):
    print("Swapping patches ...")
    adata = fbdata[64*(a-1):64*a]
    bdata = fbdata[64*(b-1):64*b]
    fbdata[64*(a-1):64*a] = bdata
    fbdata[64*(b-1):64*b] = adata
    return fbdata

def fbcopy(a, b, fbdata):
    print("Copying patch(es) ...")
    for i in a:
        adata = fbdata[64*(i-1):64*i]
        fbdata[64*(b-1):64*b] = adata
        b += 1
    return fbdata

def txedit(txdata, address, value, p=1):
    i = p - 1
    if address in range(0, 128):
        if value in range(0, 128):
            dd = txdata[128*i:128*i + 128]
            dd[address] = value
            dd = fourop.cleanvmm[dd]
            txdata[128*i:128*i + 128] = dd
    return txdata

def fbedit(fbdata, address, value, p=1):
    i = p - 1
    if address in range(0, 64):
        if value in range(0, 255):
            dd = fbdata[64:64*i + 64]
            dd[address] = value
            dd = fb01.fbclean[dd]
            fbdata[64*i:64*i + 64] = dd
    return fbdata


